<?php

/**
 * @property array $headers
 * @property array $curlOptions
 */
class RestClient
{
    public function __construct(array $headers = [], array $curlOptions = [])
    {
        $this->headers = array_merge($headers, [
            'User-agent: api-client/dynamos',
        ]);
        $this->curlOptions = $curlOptions + [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        ];
    }

    /**
     * @param string $url
     * @param array<string,mixed> $params Query params
     * @param string[] $headers
     * @return array<string,mixed>|null Json response or null on failure
     */
    public function get($url, array $params = [], array $headers = [])
    {
        $resource = curl_init();
        if (!$resource) {
            return null;
        }
        $curlOptions = $this->curlOptions + [
                CURLOPT_URL => $url . (empty($params) ? "" : "?" . http_build_query($params)),
                CURLOPT_HTTPHEADER => array_merge($this->headers, $headers),
                CURLOPT_CUSTOMREQUEST => "GET",
            ];
        return $this->send($resource, $curlOptions);
    }

    /**
     * @param string $url
     * @param array<string,mixed> $data Body
     * @param string[] $headers
     * @return array<string,mixed>|null Json response or null on failure
     */
    public function post($url, array $data = [], array $headers = [])
    {
        $resource = curl_init();
        if (!$resource) {
            return null;
        }
        $curlOptions = $this->curlOptions + [
                CURLOPT_URL => $url,
                CURLOPT_HTTPHEADER => array_merge($this->headers, $headers),
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($data),
            ];
        return $this->send($resource, $curlOptions);
    }

    /**
     * @param string $url
     * @param array<string,mixed> $data Body
     * @param string[] $headers
     * @return array<string,mixed>|null Json response or null on failure
     */
    public function put($url, array $data = [], array $headers = [])
    {
        $resource = curl_init();
        if (!$resource) {
            return null;
        }
        $curlOptions = $this->curlOptions + [
                CURLOPT_URL => $url,
                CURLOPT_HTTPHEADER => array_merge($this->headers, $headers),
                CURLOPT_CUSTOMREQUEST => "PUT",
                CURLOPT_POSTFIELDS => json_encode($data),
            ];
        return $this->send($resource, $curlOptions);
    }

    /**
     * @param string $url
     * @param array<string,mixed> $data Body
     * @param string[] $headers
     * @return array<string,mixed>|null Json response or null on failure
     */
    public function patch($url, array $data = [], array $headers = [])
    {
        $resource = curl_init();
        if (!$resource) {
            return null;
        }
        $curlOptions = $this->curlOptions + [
                CURLOPT_URL => $url,
                CURLOPT_HTTPHEADER => array_merge($this->headers, $headers),
                CURLOPT_CUSTOMREQUEST => "PATCH",
                CURLOPT_POSTFIELDS => json_encode($data),
            ];
        return $this->send($resource, $curlOptions);
    }

    /**
     * @param string $url
     * @param string[] $headers
     * @return array<string,mixed>|null Json response or null on failure
     */
    public function delete($url, array $headers = [])
    {
        $resource = curl_init();
        if (!$resource) {
            return null;
        }
        $curlOptions = $this->curlOptions + [
                CURLOPT_URL => $url,
                CURLOPT_HTTPHEADER => array_merge($this->headers, $headers),
                CURLOPT_CUSTOMREQUEST => "DELETE",
            ];
        return $this->send($resource, $curlOptions);
    }

    /**
     * @param resource $resource
     * @param array<int,mixed> $curlOptions
     * @return array<string,mixed>|null Json response or null on failure
     */
    private function send($resource, $curlOptions)
    {
        if (!curl_setopt_array($resource, $curlOptions)) {
            return null;
        }

        $response = curl_exec($resource);
        $curlInfo = curl_getinfo($resource);
        $curlErrno = curl_errno($resource);
        $curlError = curl_error($resource);

        $logDir = __DIR__ . '/../logs';

        if (!is_dir($logDir)) {
            mkdir($logDir, 0777, true);
        }

        $logFile = $logDir . '/clicksign_requests.log';

        $log = [
            'url' => isset($curlInfo['url']) ? $curlInfo['url'] : null,
            'http_code' => isset($curlInfo['http_code']) ? $curlInfo['http_code'] : null,
            'request' => isset($curlOptions[CURLOPT_POSTFIELDS]) ? $curlOptions[CURLOPT_POSTFIELDS] : null,
            'response' => $response,
            'curl_errno' => $curlErrno,
            'curl_error' => $curlError,
            'timestamp' => date('c'),
        ];

        file_put_contents($logFile, json_encode($log, JSON_UNESCAPED_SLASHES) . PHP_EOL, FILE_APPEND);

        if (!$response) {
            $error = curl_error($resource);
            curl_close($resource);
            return null;
        } else {
            $transfer = curl_getinfo($resource);
            if ($transfer['http_code'] >= 400) {
                return null;
            }
            $json = json_decode($response, true);
            curl_close($resource);
            return $json;
        }
    }
}