<?php

include __DIR__ . '/restclient.php';
include __DIR__ . '/config.php';

/**
 * @property string $baseUrl
 * @property RestClient $restClient
 */
class Clicksign
{
    /**
     * @param string|null $baseUrl
     * @param string|null $token
     * @param RestClient|null $restClient
     */
    public function __construct($baseUrl = null, $token = null, $restClient = null)
    {
        if (empty($baseUrl)) {
            global $clicksignConfig;
            $baseUrl = $clicksignConfig['baseUrl'];
        }
        if (empty($token)) {
            global $clicksignConfig;
            $token = $clicksignConfig['token'];
        }
        if (!$restClient) {
            $restClient = new RestClient([
                "Authorization: $token",
                "Content-type: application/vnd.api+json",
                "accept: application/vnd.api+json"
            ]);
        }
        $this->restClient = $restClient;
        $this->baseUrl = $baseUrl;
    }

    public function createEnvelope($name)
    {
        return $this->restClient->post("{$this->baseUrl}/envelopes", [
            'data' => [
                'type' => 'envelopes',
                'attributes' => [
                    'name' => $name,
                ],
            ],
        ]);
    }

    public function addDocumentToEnvelope($envelopeId, $filename, $content)
    {
        return $this->restClient->post("{$this->baseUrl}/envelopes/{$envelopeId}/documents", [
            'data' => [
                'type' => 'documents',
                'attributes' => [
                    'filename' => $filename,
                    'content_base64' => $content,
                ],
            ],
        ]);
    }

    public function addSignerToEnvelope($envelopeId, $name, $email)
    {
        return $this->restClient->post("{$this->baseUrl}/envelopes/$envelopeId/signers", [
            'data' => [
                'type' => 'signers',
                'attributes' => [
                    'name' => $name,
                    'email' => $email,
                ],
            ],
        ]);
    }

    public function addRequirementToEnvelope($envelopeId, $documentId, $signerId, array $requirement)
    {
        return $this->restClient->post("{$this->baseUrl}/envelopes/$envelopeId/requirements", [
            'data' => [
                'type' => 'requirements',
                'attributes' => $requirement,
                'relationships' => [
                    'document' => [
                        'data' => ['type' => 'documents', 'id' => $documentId]
                    ],
                    'signer' => [
                        'data' => ['type' => 'signers', 'id' => $signerId]
                    ],
                ]
            ],
        ]);
    }

    public function activateEnvelope($envelopeId)
    {
        return $this->restClient->patch("{$this->baseUrl}/envelopes/$envelopeId", [
            'data' => [
                'id' => $envelopeId,
                'type' => 'envelopes',
                'attributes' => [
                    'status' => 'running',
                ],
            ],
        ]);
    }

    public function cancelDocument($envelopeId, $documentId)
    {
        return $this->restClient->patch("{$this->baseUrl}/envelopes/$envelopeId/documents/$documentId", [
            'data' => [
                'id' => $documentId,
                'type' => 'documents',
                'attributes' => [
                    'status' => 'canceled',
                ],
            ],
        ]);
    }

    public function notifySigners($envelopeId)
    {
        return $this->restClient->post("{$this->baseUrl}/envelopes/$envelopeId/notifications", [
            'data' => [
                'type' => 'notifications',
                'attributes' => [
                    'message' => 'Parabéns! Agora falta pouco para concluir a aquisição do seu Lote.'
                ],
            ],
        ]);
    }
}